-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

local thMain = g_thPFConfig
THSprayerSpec = {}
THSprayerSpec.SPEC_KEY = "spec_" .. thMain.modName .. ".thSprayer"
THSprayerSpec.XML_KEY = thMain.xmlKey .. ".sprayer"
THSprayerSpecTable = {}
THExtendedSprayer = {}
local debugFlagId = THDebugUtil.createFlagId("THSprayerSpec")
THSprayerSpec.debugFlagId = debugFlagId
local function initScript()
    local pfData = thMain.precisionFarming
    if pfData.isFound then
        local ExtendedSprayer = pfData.modEnv.ExtendedSprayer
        THUtils.setFunctionHook(ExtendedSprayer, "registerOverwrittenFunctions", false, false, nil, THExtendedSprayer.gInj_registerOverwrittenFunctions)
        THUtils.setFunctionHook(ExtendedSprayer, "onChangedFillType", false, true, nil, THExtendedSprayer.gInj_onChangedFillType)
        THUtils.setFunctionHook(ExtendedSprayer, "onEndWorkAreaProcessing", false, true, nil, THExtendedSprayer.gInj_onEndWorkAreaProcessing)
    end
end
local function getSpecTable(parentEnv)
    return THUtils.call(function()
        if parentEnv ~= nil then
            return parentEnv[THSprayerSpec.SPEC_KEY]
        end
    end)
end
THSprayerSpec.getSpecTable = getSpecTable
function THSprayerSpec.prerequisitesPresent(specializations)
    if SpecializationUtil.hasSpecialization(Sprayer, specializations) then
        return true
    end
    return false
end
function THSprayerSpec.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", THSprayerSpec)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", THSprayerSpec)
end
function THSprayerSpec.registerOverwrittenFunctions(vehicleType)
    local pfData = thMain.precisionFarming
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getFillUnitAllowsFillType", THSprayerSpec.sInj_getFillUnitAllowsFillType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getSprayerUsage", THSprayerSpec.sInj_getSprayerUsage)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getExternalFill", THSprayerSpec.sInj_getExternalFill)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "updateExtendedSprayerNozzleEffectState", THSprayerSpec.sInj_updateExtendedSprayerNozzleEffectState)
    if SpecializationUtil.hasSpecialization(SowingMachine, vehicleType.specializations) then
        SpecializationUtil.registerOverwrittenFunction(vehicleType, "processSowingMachineArea", THSprayerSpec.sInj_processSowingMachineArea)
    end
    if pfData ~= nil and pfData.isFound then
        THUtils.addTask(true, function()
            local ExtendedSprayer = pfData.modEnv.ExtendedSprayer
            if SpecializationUtil.hasSpecialization(ExtendedSprayer, vehicleType.specializations) then
                SpecializationUtil.registerOverwrittenFunction(vehicleType, "getCurrentSprayerMode", THExtendedSprayer.sInj_getCurrentSprayerMode)
            end
        end)
    end
end
function THSprayerSpec.onPreLoad(self)
    THUtils.call(function()
        local specTable = getSpecTable(self)
        specTable.sprayerTypeFlags = {
            default = {
                isFertilizerSprayer = false,
                isSlurryTanker = false,
                isManureSpreader = false
            },
            previous = {}
        }
        specTable.pfSprayerTypeFlags = {
            default = {
                isSolidFertilizerSprayer = false,
                isLiquidFertilizerSprayer = false,
                isManureSpreader = false,
                isSlurryTanker = false
            },
            previous = {}
        }
        specTable.allowedSprayTypeGroups = {}
        specTable.isSolidSprayer = false
        specTable.isLiquidSprayer = false
        specTable.isSprayerTypeInitialized = false
    end)
end
function THSprayerSpec.onLoad(self)
    THUtils.call(function()
        local pfData = thMain.precisionFarming
        local specTable = getSpecTable(self)
        local sprayerSpec = self.spec_sprayer
        for sprayerKey in pairs(specTable.sprayerTypeFlags.default) do
            local isActive = sprayerSpec[sprayerKey]
            if isActive == nil then
                THUtils.errorMsg(false, "Missing sprayer specialization key: %s", sprayerKey)
            else
                specTable.sprayerTypeFlags.default[sprayerKey] = isActive
            end
        end
        if pfData.isFound then
            local ExtendedSprayer = pfData.modEnv.ExtendedSprayer
            local extendedSprayerSpec = self[ExtendedSprayer.SPEC_TABLE_NAME]
            if extendedSprayerSpec ~= nil then
                for sprayerKey in pairs(specTable.pfSprayerTypeFlags.default) do
                    local isActive = extendedSprayerSpec[sprayerKey]
                    if isActive == nil then
                        THUtils.errorMsg(false, "Missing extended sprayer specialization key: %s", sprayerKey)
                    else
                        specTable.pfSprayerTypeFlags.default[sprayerKey] = isActive
                    end
                end
            end
        end
    end)
end
function THSprayerSpec.getSprayerFillType(self)
    local sprayerSpec = self.spec_sprayer
    local pfData = thMain.precisionFarming
    if pfData.isFound then
        local ExtendedSprayer = pfData.modEnv.ExtendedSprayer
        local extendedSprayerSpec = self[ExtendedSprayer.SPEC_TABLE_NAME]
        if extendedSprayerSpec ~= nil then
            local sourceVehicle, sourceFillUnitIndex = ExtendedSprayer.getFillTypeSourceVehicle(self)
            if sourceVehicle ~= nil and sourceFillUnitIndex ~= nil then
                return sourceVehicle:getFillUnitFillType(sourceFillUnitIndex), sourceFillUnitIndex, sourceVehicle
            end
        end
    end
    if sprayerSpec ~= nil then
        local sprayerFillUnitIndex = self:getSprayerFillUnitIndex()
        if sprayerFillUnitIndex ~= nil then
            local fillUnitFillLevel = self:getFillUnitFillLevel(sprayerFillUnitIndex)
            local fillUnitFillType = self:getFillUnitFillType(sprayerFillUnitIndex)
            if fillUnitFillLevel ~= nil and fillUnitFillLevel > 0
                and fillUnitFillType ~= nil
            then
                return fillUnitFillType, sprayerFillUnitIndex, self
            end
        end
        if sprayerSpec.supportedSprayTypes ~= nil then
            for supportedIndex = 1, #sprayerSpec.supportedSprayTypes do
                local sprayTypeIndex = sprayerSpec.supportedSprayTypes[supportedIndex]
                local fillTypeSources = sprayerSpec.fillTypeSources[sprayTypeIndex]
                if fillTypeSources ~= nil then
                    for sourceIndex = 1, #fillTypeSources do
                        local sourceData = fillTypeSources[sourceIndex]
                        local sourceVehicle = sourceData.vehicle
                        if sourceVehicle ~= nil and sourceData.fillUnitIndex ~= nil
                            and sourceVehicle.getFillUnitFillLevel ~= nil
                            and sourceVehicle.getFillUnitFillType ~= nil
                        then
                            local fillUnitFillLevel = sourceVehicle:getFillUnitFillLevel(sourceData.fillUnitIndex)
                            local fillUnitFillType = sourceVehicle:getFillUnitFillType(sourceData.fillUnitIndex)
                            if fillUnitFillLevel ~= nil and fillUnitFillLevel > 0
                                and fillUnitFillType ~= nil
                            then
                                return fillUnitFillType, sourceData.fillUnitIndex, sourceVehicle
                            end
                        end
                    end
                end
            end
        end
        return FillType.UNKNOWN, sprayerFillUnitIndex, self
    end
end
function THSprayerSpec.setCurrentSprayerType(self, fillType)
    local specTable = getSpecTable(self)
    local pfData = thMain.precisionFarming
    local sprayerSpec = self.spec_sprayer
    local extendedSprayerSpec = nil
    if pfData.isFound then
        local ExtendedSprayer = pfData.modEnv.ExtendedSprayer
        extendedSprayerSpec = self[ExtendedSprayer.SPEC_TABLE_NAME]
    end
    local xtSprayTypeData = thMain:getExtendedSprayTypeByFillType(fillType)
    if xtSprayTypeData ~= nil then
        local isDebugEnabled = THDebugUtil.getIsEnabled(debugFlagId, THDebugLevel.UPDATE)
        if sprayerSpec ~= nil then
            for sprayerTypeFlagId in pairs(specTable.sprayerTypeFlags.default) do
                specTable.sprayerTypeFlags.previous[sprayerTypeFlagId] = sprayerSpec[sprayerTypeFlagId] == true
                sprayerSpec[sprayerTypeFlagId] = false
            end
        end
        if extendedSprayerSpec ~= nil then
            for sprayerTypeFlagId in pairs(specTable.pfSprayerTypeFlags.default) do
                specTable.pfSprayerTypeFlags.previous[sprayerTypeFlagId] = extendedSprayerSpec[sprayerTypeFlagId] == true
                extendedSprayerSpec[sprayerTypeFlagId] = false
            end
        end
        if xtSprayTypeData.group == THPFConfig.SPRAY_TYPE_GROUP.FERTILIZER
            or xtSprayTypeData.group == THPFConfig.SPRAY_TYPE_GROUP.LIME
        then
            if sprayerSpec ~= nil then
                sprayerSpec.isFertilizerSprayer = true
            end
            if extendedSprayerSpec ~= nil then
                if xtSprayTypeData.isLiquid then
                    extendedSprayerSpec.isLiquidFertilizerSprayer = true
                else
                    extendedSprayerSpec.isSolidFertilizerSprayer = true
                end
            end
        elseif xtSprayTypeData.group == THPFConfig.SPRAY_TYPE_GROUP.MANURE
            or xtSprayTypeData.group == THPFConfig.SPRAY_TYPE_GROUP.DIGESTATE
        then
            if sprayerSpec ~= nil then
                if xtSprayTypeData.isLiquid then
                    sprayerSpec.isSlurryTanker = true
                else
                    sprayerSpec.isManureSpreader = true
                end
            end
            if extendedSprayerSpec ~= nil then
                if xtSprayTypeData.isLiquid then
                    extendedSprayerSpec.isSlurryTanker = true
                else
                    extendedSprayerSpec.isManureSpreader = true
                end
            end
        end
        if isDebugEnabled then
            if sprayerSpec ~= nil then
                THUtils.displayMsg("Sprayer: updated type flags:")
                for sprayerTypeFlagId, defaultIsActive in pairs(specTable.sprayerTypeFlags.default) do
                    THUtils.displayMsg("- %s: %s > %s", sprayerTypeFlagId, defaultIsActive, sprayerSpec[sprayerTypeFlagId])
                end
                THUtils.displayMsg("")
            end
            if extendedSprayerSpec ~= nil then
                THUtils.displayMsg("ExtendedSprayer: updated type flags:")
                for sprayerTypeFlagId, defaultIsActive in pairs(specTable.pfSprayerTypeFlags.default) do
                    THUtils.displayMsg("- %s: %s > %s", sprayerTypeFlagId, defaultIsActive, extendedSprayerSpec[sprayerTypeFlagId])
                end
                THUtils.displayMsg("")
            end
        end
    else
        THSprayerSpec.resetSprayerTypes(self)
    end
end
function THSprayerSpec.resetSprayerTypes(self)
    local specTable = getSpecTable(self)
    local pfData = thMain.precisionFarming
    local sprayerSpec = self.spec_sprayer
    local extendedSprayerSpec = nil
    if pfData.isFound then
        local ExtendedSprayer = pfData.modEnv.ExtendedSprayer
        extendedSprayerSpec = self[ExtendedSprayer.SPEC_TABLE_NAME]
    end
    local isDebugEnabled = THDebugUtil.getIsEnabled(debugFlagId, THDebugLevel.UPDATE)
    local isSprayerReset, isExtendedSprayerReset = false, false
    if sprayerSpec ~= nil then
        for sprayerTypeFlagId, defaultIsActive in pairs(specTable.sprayerTypeFlags.default) do
            local lastIsActive = sprayerSpec[sprayerTypeFlagId]
            local prevIsActive = specTable.sprayerTypeFlags.previous[sprayerTypeFlagId]
            if prevIsActive ~= nil then
                sprayerSpec[sprayerTypeFlagId] = prevIsActive
                specTable.sprayerTypeFlags.previous[sprayerTypeFlagId] = nil
            else
                sprayerSpec[sprayerTypeFlagId] = defaultIsActive == true
            end
            if sprayerSpec[sprayerTypeFlagId] ~= lastIsActive then
                if not isSprayerReset then
                    if isDebugEnabled then
                        THUtils.displayMsg("Sprayer: resetting type flags:")
                    end
                    isSprayerReset = true
                end
                if isDebugEnabled then
                    THUtils.displayMsg("- %s: %s > %s", sprayerTypeFlagId, lastIsActive, sprayerSpec[sprayerTypeFlagId])
                end
            end
        end
    end
    if extendedSprayerSpec ~= nil then
        for sprayerTypeFlagId, defaultIsActive in pairs(specTable.pfSprayerTypeFlags.default) do
            local lastIsActive = extendedSprayerSpec[sprayerTypeFlagId]
            local prevIsActive = specTable.pfSprayerTypeFlags.previous[sprayerTypeFlagId]
            if prevIsActive ~= nil then
                extendedSprayerSpec[sprayerTypeFlagId] = prevIsActive
                specTable.pfSprayerTypeFlags.previous[sprayerTypeFlagId] = nil
            else
                extendedSprayerSpec[sprayerTypeFlagId] = defaultIsActive == true
            end
            if extendedSprayerSpec[sprayerTypeFlagId] ~= lastIsActive then
                if not isExtendedSprayerReset then
                    if isDebugEnabled then
                        THUtils.displayMsg("ExtendedSprayer: resetting type flags:")
                    end
                    isExtendedSprayerReset = true
                end
                if isDebugEnabled then
                    THUtils.displayMsg("- %s: %s > %s", sprayerTypeFlagId, lastIsActive, extendedSprayerSpec[sprayerTypeFlagId])
                end
            end
        end
    end
    if isDebugEnabled and (isSprayerReset or isExtendedSprayerReset) then
        THUtils.displayMsg("")
    end
end
function THSprayerSpec.mapExtendedSprayTypeConstants(self, fillType, mapSprayType, mapFillType, callingFuncName)
    callingFuncName = THUtils.validateArg(type(callingFuncName) == THValueType.STRING and callingFuncName ~= "", "callingFuncName", callingFuncName, ">> error <<")
    local xtSprayTypeData = thMain:getExtendedSprayTypeByFillType(fillType)
    if xtSprayTypeData ~= nil then
        local isSprayTypeMapped, isFillTypeMapped = xtSprayTypeData:activateMapping(mapSprayType, mapFillType)
        if self:getIsActive() then
            if THDebugUtil.getIsEnabled(debugFlagId, THDebugLevel.UPDATE) then
                if isSprayTypeMapped then
                    THUtils.displayMsg("THSprayerSpec (%s): Spray type %q mapped to %q", callingFuncName, xtSprayTypeData.baseId, xtSprayTypeData.id)
                end
                if isFillTypeMapped then
                    THUtils.displayMsg("THSprayerSpec (%s): FillType %q mapped to %q", callingFuncName, xtSprayTypeData.baseFillTypeId, xtSprayTypeData.fillTypeId)
                end
                if isSprayTypeMapped or isFillTypeMapped then
                    THUtils.displayMsg("")
                end
            end
        end
        return xtSprayTypeData, isSprayTypeMapped, isFillTypeMapped
    end
    return nil, false, false
end
function THSprayerSpec.sInj_getFillUnitAllowsFillType(self, superFunc, fillUnitIndex, fillTypeIndex, ...)
    local specTable = getSpecTable(self)
    local function appendFunc(rSuccess, ...)
        if fillUnitIndex ~= nil and fillTypeIndex ~= nil then
            THUtils.call(function()
                local xtSprayTypeData = thMain:getExtendedSprayTypeByFillType(fillTypeIndex)
                if xtSprayTypeData ~= nil then
                    if not specTable.isSprayerTypeInitialized then
                        local xtSprayTypes, xtNumSprayTypes = thMain:getExtendedSprayTypes()
                        if xtSprayTypes ~= nil and xtNumSprayTypes > 0 then
                            for xtSprayIndex = 1, xtNumSprayTypes do
                                local xtOtherSprayTypeData = xtSprayTypes[xtSprayIndex]
                                if superFunc(self, fillUnitIndex, xtOtherSprayTypeData.fillTypeIndex) then
                                    if xtOtherSprayTypeData.isLiquid then
                                        specTable.isLiquidSprayer = true
                                    else
                                        specTable.isSolidSprayer = true
                                    end
                                    specTable.allowedSprayTypeGroups[xtOtherSprayTypeData.group] = true
                                end
                            end
                        end
                        if THDebugUtil.getIsEnabled(thMain.debugFlagId) then
                            local vehicleName = self:getName() or "THSprayerSpec"
                            THUtils.displayMsg("%s: sprayer type initialized:", vehicleName)
                            THUtils.displayMsg("- isLiquidSprayer: %s", specTable.isLiquidSprayer)
                            THUtils.displayMsg("- isSolidSprayer: %s", specTable.isSolidSprayer)
                            THUtils.displayMsg("- allowed spray type groups:")
                            THDebugUtil.printTable(specTable.allowedSprayTypeGroups)
                        end
                        specTable.isSprayerTypeInitialized = true
                    end
                end
            end)
        end
        return rSuccess, ...
    end
    return appendFunc(superFunc(self, fillUnitIndex, fillTypeIndex, ...))
end
function THSprayerSpec.sInj_getSprayerUsage(self, superFunc, fillTypeIndex, ...)
    local pfData = thMain.precisionFarming
    local lastPHMapUsageSprayType, isPHMapUsageSet = nil, false
    if fillTypeIndex ~= nil then
        THUtils.call(function()
            local xtSprayTypeData = thMain:getExtendedSprayTypeByFillType(fillTypeIndex)
            if xtSprayTypeData ~= nil then
                if xtSprayTypeData.group == THPFConfig.SPRAY_TYPE_GROUP.LIME then
                    if pfData.isFound then
                        local ExtendedSprayer = pfData.modEnv.ExtendedSprayer
                        local extendedSprayerSpec = self[ExtendedSprayer.SPEC_TABLE_NAME]
                        if extendedSprayerSpec ~= nil then
                            local thPHMapData = THPHMap.getCustomData(extendedSprayerSpec.pHMap)
                            if thPHMapData ~= nil then
                                lastPHMapUsageSprayType = thPHMapData:setUsageSprayType(xtSprayTypeData.sprayTypeIndex)
                                isPHMapUsageSet = true
                            end
                        end
                    end
                end
            end
        end)
    end
    local function appendFunc(...)
        THUtils.call(function()
            if isPHMapUsageSet then
                local ExtendedSprayer = pfData.modEnv.ExtendedSprayer
                local extendedSprayerSpec = self[ExtendedSprayer.SPEC_TABLE_NAME]
                local thPHMapData = THPHMap.getCustomData(extendedSprayerSpec.pHMap)
                if thPHMapData ~= nil then
                    thPHMapData:setUsageSprayType(lastPHMapUsageSprayType)
                end
            end
        end)
        return ...
    end
    return appendFunc(superFunc(self, fillTypeIndex, ...))
end
function THSprayerSpec.sInj_getExternalFill(self, superFunc, fillTypeIndex, dt, ...)
    if fillTypeIndex ~= nil then
        THUtils.call(function()
            THSprayerSpec.mapExtendedSprayTypeConstants(self, fillTypeIndex, true, true, "getExternalFill")
        end)
    end
    local function appendFunc(...)
        THUtils.call(function()
            thMain:resetExtendedSprayTypeConstants()
        end)
        return ...
    end
    return appendFunc(superFunc(self, fillTypeIndex, dt, ...))
end
function THSprayerSpec.sInj_processSowingMachineArea(self, superFunc, ...)
    local sprayerSpec = self.spec_sprayer
    if sprayerSpec ~= nil then
        THUtils.call(function()
            local sprayFillType = sprayerSpec.workAreaParameters.sprayFillType
            if sprayFillType ~= nil and sprayFillType ~= FillType.UNKNOWN then
                THSprayerSpec.mapExtendedSprayTypeConstants(self, sprayFillType, true, true, "processSowingMachineArea")
            end
        end)
    end
    local function appendFunc(...)
        if sprayerSpec ~= nil then
            THUtils.call(function()
                thMain:resetExtendedSprayTypeConstants()
            end)
        end
        return ...
    end
    return appendFunc(superFunc(self, ...))
end
function THSprayerSpec.sInj_updateExtendedSprayerNozzleEffectState(self, superFunc, ...)
    local sprayerSpec = self.spec_sprayer
    if sprayerSpec ~= nil then
        THUtils.call(function()
            local sprayFillType = sprayerSpec.workAreaParameters.sprayFillType
            if sprayFillType ~= nil and sprayFillType ~= FillType.UNKNOWN then
                THSprayerSpec.mapExtendedSprayTypeConstants(self, sprayFillType, true, true, "updateExtendedSprayerNozzleEffectState")
            end
        end)
    end
    local function appendFunc(...)
        if sprayerSpec ~= nil then
            THUtils.call(function()
                thMain:resetExtendedSprayTypeConstants()
            end)
        end
        return ...
    end
    return appendFunc(superFunc(self, ...))
end
function THExtendedSprayer.sInj_getCurrentSprayerMode(self, superFunc, ...)
    local pfData = thMain.precisionFarming
    THUtils.call(function()
        if pfData.isFound then
            local ExtendedSprayer = pfData.modEnv.ExtendedSprayer
            local spraySourceVehicle, sourceFillUnitIndex = ExtendedSprayer.getFillTypeSourceVehicle(self)
            if spraySourceVehicle ~= nil and sourceFillUnitIndex ~= nil then
                local sourceFillType = spraySourceVehicle:getFillUnitLastValidFillType(sourceFillUnitIndex)
                if sourceFillType ~= nil and sourceFillType ~= FillType.UNKNOWN then
                    THSprayerSpec.mapExtendedSprayTypeConstants(self, sourceFillType, true, true, "getCurrentSprayerMode")
                end
            end
        end
    end)
    local function appendFunc(...)
        THUtils.call(function()
            thMain:resetExtendedSprayTypeConstants()
        end)
        return ...
    end
    return appendFunc(superFunc(self, ...))
end
function THExtendedSprayer.gInj_registerOverwrittenFunctions(superFunc, vehicleType, ...)
    local function appendFunc(...)
        THUtils.call(function()
            SpecializationUtil.registerOverwrittenFunction(vehicleType, "getCurrentSprayerMode", THExtendedSprayer.sInj_getCurrentSprayerMode)
        end)
        return ...
    end
    return appendFunc(superFunc(vehicleType, ...))
end
function THExtendedSprayer.gInj_onChangedFillType(self, superFunc, arg2, fillTypeIndex, ...)
    THUtils.call(function()
        local xtSprayTypeData = thMain:getExtendedSprayTypeByFillType(fillTypeIndex)
        local pfData = thMain.precisionFarming
        if xtSprayTypeData ~= nil then
            THSprayerSpec.mapExtendedSprayTypeConstants(self, xtSprayTypeData.fillTypeIndex, true, true, "onChangedFillType")
            THSprayerSpec.setCurrentSprayerType(self, xtSprayTypeData.fillTypeIndex)
            if xtSprayTypeData.group == THPFConfig.SPRAY_TYPE_GROUP.LIME then
                if pfData.isFound then
                    local ExtendedSprayer = pfData.modEnv.ExtendedSprayer
                    local extendedSprayerSpec = self[ExtendedSprayer.SPEC_TABLE_NAME]
                    if extendedSprayerSpec ~= nil then
                        extendedSprayerSpec.isSolidFertilizerSprayer = true
                    end
                end
            end
        end
    end)
    local function appendFunc(...)
        THUtils.call(function()
            thMain:resetExtendedSprayTypeConstants()
            THSprayerSpec.resetSprayerTypes(self)
        end)
        return ...
    end
    return appendFunc(superFunc(self, arg2, fillTypeIndex, ...))
end
function THExtendedSprayer.gInj_onEndWorkAreaProcessing(self, superFunc, ...)
    local sprayerSpec = self.spec_sprayer
    THUtils.call(function()
        if sprayerSpec ~= nil then
            local sprayFillType = sprayerSpec.workAreaParameters.sprayFillType
            if sprayFillType ~= nil and sprayFillType ~= FillType.UNKNOWN then
                THSprayerSpec.mapExtendedSprayTypeConstants(self, sprayFillType, true, true, "onEndWorkAreaProcessing")
            end
        end
    end)
    local function appendFunc(...)
        THUtils.call(function()
            thMain:resetExtendedSprayTypeConstants()
        end)
        return ...
    end
    return appendFunc(superFunc(self, ...))
end
THUtils.call(initScript)